#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

typedef struct {
    long long x;
    long long y;
    int index;
    double angle;
} Vector;

long long squared_distance(long long x, long long y) {
    return x * x + y * y;
}

int compare_vectors(const void *a, const void *b) {
    Vector *vecA = (Vector *)a;
    Vector *vecB = (Vector *)b;
    if (vecA->angle < vecB->angle) return -1;
    if (vecA->angle > vecB->angle) return 1;
    return 0;
}

void solve() {
    int n;
    if (scanf("%d", &n) != 1) return;

    Vector *vectors = (Vector *)malloc(n * sizeof(Vector));
    if (vectors == NULL) {
        perror("Failed to allocate memory for vectors");
        return;
    }

    for (int i = 0; i < n; i++) {
        long long x, y;
        if (scanf("%lld %lld", &x, &y) != 2) {
            free(vectors);
            return;
        }
        vectors[i].x = x;
        vectors[i].y = y;
        vectors[i].index = i + 1;
        vectors[i].angle = atan2((double)y, (double)x);
        if (vectors[i].angle < 0) {
            vectors[i].angle += 2 * M_PI;
        }
    }

    qsort(vectors, n, sizeof(Vector), compare_vectors);

    Vector *doubled_vectors = (Vector *)malloc(2 * n * sizeof(Vector));
    if (doubled_vectors == NULL) {
        perror("Failed to allocate memory for doubled vectors");
        free(vectors);
        return;
    }
    
    // Copy the sorted vectors
    memcpy(doubled_vectors, vectors, n * sizeof(Vector));
    
    // Copy the vectors again, adjusting the angle by 2*PI for the wrap-around
    for (int i = 0; i < n; i++) {
        doubled_vectors[n + i].x = vectors[i].x;
        doubled_vectors[n + i].y = vectors[i].y;
        doubled_vectors[n + i].index = vectors[i].index;
        doubled_vectors[n + i].angle = vectors[i].angle + 2 * M_PI;
    }
    free(vectors);

    long long max_dist_sq = 0;
    int best_start = -1;
    int best_end = -1;

    long long current_sum_x = 0;
    long long current_sum_y = 0;
    int start = 0;

    for (int end = 0; end < 2 * n; end++) {
        current_sum_x += doubled_vectors[end].x;
        current_sum_y += doubled_vectors[end].y;

        // The condition to shrink the window: angle_span > PI
        // The angle span is simply doubled_vectors[end].angle - doubled_vectors[start].angle
        while (start <= end && doubled_vectors[end].angle - doubled_vectors[start].angle > M_PI + 1e-9) {
            current_sum_x -= doubled_vectors[start].x;
            current_sum_y -= doubled_vectors[start].y;
            start++;
        }

        long long dist_sq = squared_distance(current_sum_x, current_sum_y);
        if (dist_sq > max_dist_sq) {
            max_dist_sq = dist_sq;
            best_start = start;
            best_end = end;
        }
    }

    if (best_start == -1) {
        printf("0\n\n");
    } else {
        int k = best_end - best_start + 1;
        printf("%d\n", k);
        for (int i = best_start; i <= best_end; i++) {
            printf("%d%s", doubled_vectors[i].index, (i == best_end) ? "" : " ");
        }
        printf("\n");
    }

    free(doubled_vectors);
}

int main() {
    #ifndef M_PI
        #define M_PI 3.14159265358979323846
    #endif
    solve();
    return 0;
}